/* vim:set syntax=c nu si background=dark: */
/*********************************************
* SPI to MAX7221 (using spidev driver)
*********************************************/

#include <math.h>
#include <stdint.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <linux/types.h>
#include <linux/spi/spidev.h>

void timestamp(const char *s)
{
    time_t curtime;
    struct tm *loctime;
    char timestr[20];

    curtime = time(NULL);
    loctime = localtime(&curtime);
    strftime(timestr, 20, "%H:%M.%S", loctime);

    printf("%s - %s", timestr, s);
}


static void pabort(const char *s)
{
    perror(s);
    abort();
}

static const char *device = "/dev/spidev0.0";
static uint8_t mode;
static uint8_t bits = 8;
static uint32_t speed = 200000;
static uint16_t delay;

static void matrixwrite(int fd, unsigned char max_address, unsigned char max_data)
{
    uint8_t tx[] = { max_address, max_data, };
    write(fd, tx, 2);
}

uint8_t txdata[72] = {
//      rrrrrrr-gggggggg--bbbbbbb/rrrrrrrr-ggggggggg-bbbbbbb/
	0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0xff, 0x00, 0x00,	// right
	0xff, 0xf0, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0x00,	//  ^
	0xff, 0xf0, 0x00, 0xff, 0x00, 0x00, 0x00, 0x0f, 0xff,	//  |
	0x00, 0x0f, 0xff, 0x00, 0xff, 0xf0, 0x00, 0x00, 0x00,	// left

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// right
	0x00, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	//  ^
	0x00, 0xf0, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x0f, 0x00,	//  |
	0x00, 0x0f, 0xff, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff	// left
};

static void write36(int fd, int led, unsigned char r, unsigned char g, unsigned char b)
{
    write(fd, txdata, 72);
}

static void setled(int led, uint16_t r, uint16_t g, uint16_t b) {
	uint8_t *offset;

	if (led>15)
		return;

	if ((led & 0x1) == 0) {
		// even leds
		offset=txdata + (led>>1)*9;
		*offset++ = r>>4;
		*offset++ = ((r & 0xf)<<4) | ((g>>8) & 0x0f );
		*offset++ = g & 0xff;
		*offset++ = b >> 4;
		*offset   = ((b & 0xf)<<4) | ((*offset)&0xf)  ;
	} else {
		offset=txdata + (led>>1)*9 + 4; /// +4.5 *led
		*offset = (*offset)&0xf0 | ((r>>8)&0x0f);
		offset++;
		*offset++ = r & 0xff;
		*offset++ = g>>4;
		*offset++ = ((g & 0xf)<<4) | ((b>>8) & 0x0f );
		*offset++ = b & 0xff;
	}

}

static void inc36(int fd, int led, unsigned char r, unsigned char g, unsigned char b)
{
	int i;
	uint8_t *p;

	for (i=0, p=txdata; i<36; i++, p++) {
		*p=(*p)++ %256;
	}
    write(fd, txdata, 36);
}

static void initspi(int fd)
{
    int ret = 0;

/** spi mode **/
    ret = ioctl(fd, SPI_IOC_WR_MODE, &mode);
    if (ret == -1)
	pabort("can't set spi mode");

    ret = ioctl(fd, SPI_IOC_RD_MODE, &mode);
    if (ret == -1)
	pabort("can't get spi mode");

/** bits per word **/
    ret = ioctl(fd, SPI_IOC_WR_BITS_PER_WORD, &bits);
    if (ret == -1)
	pabort("can't set bits per word");

    ret = ioctl(fd, SPI_IOC_RD_BITS_PER_WORD, &bits);
    if (ret == -1)
	pabort("can't get bits per word");

/** max speed hz **/
    ret = ioctl(fd, SPI_IOC_WR_MAX_SPEED_HZ, &speed);
    if (ret == -1)
	pabort("can't set max speed hz");

    ret = ioctl(fd, SPI_IOC_RD_MAX_SPEED_HZ, &speed);
    if (ret == -1)
	pabort("can't get max speed hz");

}


void matrix_single(int fd, int delay, int repeat)
{
    uint8_t col;		// Column
    uint8_t row;		// Row
    uint8_t z;

    timestamp("2. single leds.\n");
    for (z = 1; z <= repeat; z++) {
	for (col = 1; col <= 8; col++) {
	    for (row = 0; row <= 7; row++) {
		// z = pow(2, row);
		z = 1<<row;
		matrixwrite(fd, col, z);
		usleep(delay);
	    }
	    matrixwrite(fd, col, 0);
	}
    }
}

void matrix_colwipe(int fd, int delay, int repeat)
{
    uint8_t col;		// Column
    uint8_t row;		// Row
    uint8_t z;

    timestamp("3. Column wipe.\n");
    for (z = 1; z <= repeat; z++) {
	for (col = 1; col <= 8; col++) {
	    matrixwrite(fd, col, 0xFF);
	    usleep(delay);
	}
	for (col = 1; col <= 8; col++) {
	    matrixwrite(fd, col, 0x00);
	    usleep(delay);
	}
    }
    for (z = 1; z <= repeat; z++) {
	for (col = 8; col >= 1; col--) {
	    matrixwrite(fd, col, 0xFF);
	    usleep(delay);
	}
	for (col = 1; col <= 8; col++) {
	    matrixwrite(fd, col, 0x00);
	    usleep(delay);
	}
    }
}


void matrix_linewipe(int fd, int delay, int repeat)
{
    uint8_t col;		// Column
    uint8_t row;		// Row
    uint8_t z;

    timestamp("4. Line wipe.\n");
    for (z = 1; z <= repeat; z++) {
	for (row = 0; row <= 7; row++) {
	    // z = pow(2, row);
	    z = 1<<row;
	    for (col = 1; col <= 8; col++) {
		matrixwrite(fd, col, z);
		usleep(delay);
	    }
	    for (col = 8; col >= 1; col--) {
		matrixwrite(fd, col, 0);
		usleep(delay);
	    }
	}
    }
}

void matrix_linewipeupdown(int fd, int delay, int repeat)
{
    uint8_t col;		// Column
    uint8_t row;		// Row
    uint8_t z;

    timestamp("5. Line up/down.\n");
    for (z = 1; z <= repeat; z++) {
	for (col = 1; col <= 8; col++) {
	    matrixwrite(fd, col, 0xFF);
	    usleep(delay);
	    matrixwrite(fd, col, 0);
	}
	for (col = 8; col >= 1; col--) {
	    matrixwrite(fd, col, 0xFF);
	    usleep(delay);
	    matrixwrite(fd, col, 0);
	}
    }
}


void matrix_flash(int fd, int delay1, int delay2, int repeat)
{
    uint8_t col;		// Column
    uint8_t row;		// Row
    uint8_t z;

    timestamp("6. Flash.\n");

    for (z = 1; z <= repeat; z++) {
	for (col = 1; col <= 8; col++) {
	    matrixwrite(fd, col, 0xFF);
	}
	usleep(delay1);
	for (col = 1; col <= 8; col++) {
	    matrixwrite(fd, col, 0x00);
	}
	usleep(delay2);
    }

}

void matrix_digits(int fd, int delay1, int delay2, int repeat)
{
    uint8_t col;		// Column
    uint8_t row;		// Row
    uint8_t z;

    timestamp("7. Digits.\n");

    matrixwrite(fd, 0x09, 0xFF);	// Decode mode (on)

    for (z = 1; z <= repeat; z++) {
	for (col = 1; col <= 8; col++) {
	    matrixwrite(fd, col, (col+z)%16);
	}
	usleep(delay1);
//	for (col = 1; col <= 8; col++) {
//	    matrixwrite(fd, col, 0x00);
//	}
	usleep(delay2);
    }

}

int main(int argc, char *argv[])
{
    int fd;

    fd = open(device, O_RDWR);

    if (fd < 0)
	pabort("can't open device");

    initspi(fd);

    printf("---------------------------------------\n");
    printf("spi mode: %d\n", mode);
    printf("bits per word: %d\n", bits);
    printf("max speed: %d Hz (%d KHz)\n", speed, speed / 1000);
    printf("---------------------------------------\n");

    uint8_t col;		// Column

#if 0
//All leds off.
    for (col = 1; col <= 8; col++) {
	matrixwrite(fd, col, 0);
    }

    timestamp("1. Self test.\n");
    matrixwrite(fd, 0x0F, 0x01);
    usleep(500000);
    matrixwrite(fd, 0x0F, 0x00);
    usleep(500000);

// Initialize Matrix
    matrixwrite(fd, 0x0C, 0x01);	// Normal operation
    matrixwrite(fd, 0x0B, 0x07);	// Scan Limit (all digits)
    matrixwrite(fd, 0x0A, 0x07);	// Intensity
    matrixwrite(fd, 0x09, 0x00);	// Decode mode (off)

    matrix_single(fd,  40000, 2);
    usleep(1000000);
    matrix_colwipe(fd, 100000, 5);
    usleep(1000000);
    matrix_linewipe(fd, 100000, 5);
    usleep(1000000);
    matrix_linewipeupdown(fd, 100000, 5);
    usleep(1000000);
    matrix_flash(fd, 300000, 200000, 20);
    usleep(1000000);
    matrix_digits(fd, 300000, 200000, 20);

    timestamp("Completed.\n");

//All leds off.
    for (col = 1; col <= 8; col++) {
	matrixwrite(fd, col, 0);
    }
#endif

    {
	int x;
	int i;

	int a, b, c, d;
	int run=0;
    write36(fd, 0, 300, 0, 0);
	usleep(1000000);



//	for (x=1; x<10000; x++) {
//		inc36(fd,0,0,0,0);
//		usleep(1000);
//	}

setled(0,0,0,0);
setled(1,0,0,0);
setled(2,0,0,0);
setled(3,0,0,0);
setled(4,0,0,0);
setled(5,0,0,0);
setled(6,0,0,0);
setled(7,0,0,0);
	usleep(100000);

run=1;

while(1) {
	for (i=14; i>=0; i-=2) {

	    for (x=1; x< 0x1ff; x+=4) {

		if (i&2) {a=0xfff;} else a=0;
		if (i&4) {b=0xfff;} else b=0;
		if (i&8) {c=0xfff;} else c=0;
		if (run&1) {d=0xfff;} else d=0;

		if (run%2) {
		setled(i%8,       x<<3, 0, 0);
		setled((i+2)%8, 0xfff-(x<<3), 0, 0);
		}

		if (run%3) {
		setled((i+1)%8, 0, x<<3, 0);
		setled((i+3)%8, 0, 0xfff-(x<<3), 0);
		}

		if (run%5) {
		setled((i+4)%8, 0, 0, x<<3);
		setled((i+6)%8, 0, 0, 0xfff-(x<<3));
		}

		if (run%13 == 0) {
		setled((i+5)%8, 0, x<<2, x<<2);
		setled((i+7)%8, 0, 0xfff-(x<<2), 0xfff-(x<<2));
		}

		if (run%11 == 0) {
		setled((i+5)%8, x<<3, x<<3, 0 );
		setled((i+7)%8, 0xfff-(x<<3), 0xfff-(x<<3), 0 );
		}

		if (run%7 == 0) {
		setled((i+5)%8, x<<3, x<<3, x<<3);
		setled((i+7)%8, 0xfff-(x<<3), 0xfff-(x<<3), 0xfff-(x<<3));
		}

		if (i%4) {
			setled(12,       (0xfff-x<<3)& (b^a), 0, (0xfff-x<<3)&(c^a));
			setled(15,       b&(x<<3), c&(x<<3),  d&(x<<3));
			setled(14,       d&(x<<3), b&(x<<2), x<<3);
		} else {
			setled(12,       (x<<3)& (b^a), 0, (x<<3)&(c^a));
			setled(15,       b&(0xfff-x<<3), c&(0xfff-x<<3),  d&((0xfff-x<<3)));
			setled(14,       d&(x<<3), b&(0xfff-x<<2), 0xfff-x<<3);
		}

		write36(fd, 0, 0, 0, 0);

		usleep(20000);
	    }
	setled(i,0,0,0);
	setled((i+5)%8,0,0,0);

	write36(fd, 0, 0, 0, 0);

	}

	setled(8,0x2ff,0,0);
	write36(fd, 0, 0, 0, 0);
	sleep(2);

	setled(8,0,0,0x2ff);
	write36(fd, 0, 0, 0, 0);
	run++;
    }
    close(fd);

    return;
}
 }

